/*-----------------------------------------------------------------------------

 1999, Steinberg Soft und Hardware GmbH, All Rights Reserved

-----------------------------------------------------------------------------*/

#ifndef __VstXSynth__
#include "vstxsynth.h"
#endif

enum
{
	kNumFrequencies = 128,	// 128 midi notes
	kWaveSize = 4096		// samples
};

const double midiScaler = (1. / 127.);
static float sawtooth[kWaveSize];
static float pulse[kWaveSize];
static float freqtab[kNumFrequencies];

//-----------------------------------------------------------------------------------------
//-----------------------------------------------------------------------------------------
void VstXSynth::setSampleRate (float sampleRate)
{
	AudioEffectX::setSampleRate (sampleRate);
	fScaler = (float)((double)kWaveSize / (double)sampleRate);
}

//-----------------------------------------------------------------------------------------
void VstXSynth::setBlockSize (long blockSize)
{
	AudioEffectX::setBlockSize (blockSize);
}

//-----------------------------------------------------------------------------------------
void VstXSynth::resume ()
{
	wantEvents ();
}

//-----------------------------------------------------------------------------------------
void VstXSynth::initProcess ()
{
	fPhase1 = fPhase2 = fPhase3 = 0.f;
	fScaler = (float)((double)kWaveSize / 44100.);	// we don't know the sample rate yet
	noteIsOn = false;
	currentDelta = 0;
	long i;

	// make waveforms
	long wh = kWaveSize / 4;	// 1:3 pulse
	for (i = 0; i < kWaveSize; i++)
	{
		sawtooth[i] = (float)(-1. + (2. * ((double)i / (double)kWaveSize)));
		pulse[i] = (i < wh) ? -1.f : 1.f;
	}
	// make frequency (Hz) table
	double k = 1.059463094359;	// 12th root of 2
	double a = 6.875;	// a
	a *= k;	// b
	a *= k;	// bb
	a *= k;	// c, frequency of midi note 0
	for (i = 0; i < kNumFrequencies; i++)	// 128 midi notes
	{
		freqtab[i] = (float)a;
		a *= k;
	}
}

//-----------------------------------------------------------------------------------------
void VstXSynth::process (float **inputs, float **outputs, long sampleFrames)
{
	if (noteIsOn)
	{
		float* out1 = outputs[0];
		float* out2 = outputs[1];
		float* out3 = outputs[2];
		float baseFreq = freqtab[currentNote & 0x7f] * fScaler;
		float freq1 = baseFreq + fFreq1;
		float freq2 = baseFreq + fFreq2;
		float freq3 = baseFreq + fFreq3;
		float* wave1 = (fWaveform1 < .5) ? sawtooth : pulse;
		float* wave2 = (fWaveform2 < .5) ? sawtooth : pulse;
		float* wave3 = (fWaveform3 < .5) ? sawtooth : pulse;
		float wsf = (float)kWaveSize;
		float vol = (float)(fVolume * (double)currentVelocity * midiScaler);
		
		if (currentDelta >= 0)
		{
			if (currentDelta >= sampleFrames)	// must never happen
				currentDelta = 0;
			out1 += currentDelta;
			out2 += currentDelta;
			out3 += currentDelta;
			sampleFrames -= currentDelta;
			currentDelta = 0;
		}
		// loop
		while (--sampleFrames >= 0)
		{
			(*out1++) += wave1[(long)fPhase1] * fVolume1 * vol;
			(*out2++) += wave2[(long)fPhase2] * fVolume2 * vol;
			(*out3++) += wave3[(long)fPhase3] * fVolume3 * vol;
			fPhase1 += freq1;
			while (fPhase1 >= wsf)
				fPhase1 -= wsf;
			fPhase2 += freq2;
			while (fPhase2 >= wsf)
				fPhase2 -= wsf;
			fPhase3 += freq3;
			while (fPhase3 >= wsf)
				fPhase3 -= wsf;
		}
	}						
}

//-----------------------------------------------------------------------------------------
void VstXSynth::processReplacing (float **inputs, float **outputs, long sampleFrames)
{
	float* out1 = outputs[0];
	float* out2 = outputs[1];
	float* out3 = outputs[2];
	if (noteIsOn)
	{
		float baseFreq = freqtab[currentNote & 0x7f] * fScaler;
		float freq1 = baseFreq + fFreq1;
		float freq2 = baseFreq + fFreq2;
		float freq3 = baseFreq + fFreq3;
		float* wave1 = (fWaveform1 < .5) ? sawtooth : pulse;
		float* wave2 = (fWaveform2 < .5) ? sawtooth : pulse;
		float* wave3 = (fWaveform3 < .5) ? sawtooth : pulse;
		float wsf = (float)kWaveSize;
		float vol = (float)(fVolume * (double)currentVelocity * midiScaler);
		
		if (currentDelta >= 0)
		{
			if (currentDelta >= sampleFrames)	// must never happen
				currentDelta = 0;
			else
			{
				for (long i = 0; i < currentDelta; i++)
				{
					*out1++ = 0;
					*out2++ = 0;
					*out3++ = 0;
				}
			}
			sampleFrames -= currentDelta;
			currentDelta = 0;
		}
		// loop
		while (--sampleFrames >= 0)
		{
			(*out1++) = wave1[(long)fPhase1] * fVolume1 * vol;
			(*out2++) = wave2[(long)fPhase2] * fVolume2 * vol;
			(*out3++) = wave3[(long)fPhase3] * fVolume3 * vol;
			fPhase1 += freq1;
			while (fPhase1 >= wsf)
				fPhase1 -= wsf;
			fPhase2 += freq2;
			while (fPhase2 >= wsf)
				fPhase2 -= wsf;
			fPhase3 += freq3;
			while (fPhase3 >= wsf)
				fPhase3 -= wsf;
		}
	}
	else
	{
		while (--sampleFrames >= 0)
		{
			*out1++ = 0;
			*out2++ = 0;
			*out3++ = 0;
		}
	}
}

//-----------------------------------------------------------------------------------------
long VstXSynth::processEvents (VstEvents* ev)
{
	for (long i = 0; i < ev->numEvents; i++)
	{
		if ((ev->events[i])->type != kVstMidiType)
			continue;
		VstMidiEvent* event = (VstMidiEvent*)ev->events[i];
		char* midiData = event->midiData;
		long status = midiData[0] & 0xf0;		// ignoring channel
		if (status == 0x90 || status == 0x80)	// we only look at notes
		{
			long note = midiData[1] & 0x7f;
			long velocity = midiData[2] & 0x7f;
			if (status == 0x80)
				velocity = 0;
			if (!velocity && (note == currentNote))
				noteIsOn = false;	// note off by velocity 0
			else
				noteOn (note, velocity, event->deltaFrames);
		}
		else if (status == 0xb0 && midiData[1] == 0x7e)	// all notes off
			noteIsOn = false;
		event++;
	}
	return 1;	// want more
}

//-----------------------------------------------------------------------------------------
void VstXSynth::noteOn (long note, long velocity, long delta)
{
	currentNote = note;
	currentVelocity = velocity;
	currentDelta = delta;
	noteIsOn = true;
}
